#ifndef KUMARMEDIAN_H
#define KUMARMEDIAN_H

#include <vector>
#include <memory>
#include <functional>
#include <cmath>

#include "Sampling.cpp"
#include "Point.hpp"
#include "Metric.hpp"
#include "Norm.hpp"
#include "EuclideanSpace.hpp"
#include "KMedian.hpp"

/**
 * @brief 1-median approximation
 * 
 * Kumar, Sabharwal, Sen: Linear-time approximation schemes for clustering problems in any dimensions
 */
class KumarMedian
{
public:
	KumarMedian(std::function<Metric<Point>*()> createMetric, std::function<Norm<Point>*()> createNorm);
	
	/**
	 * @brief Approximate k-median (choose best out of n)
	 * @param begin Point iterator
	 * @param end Point iterator
	 * @param eps Factor of approximation
	 * @param rounds Number of rounds
	 */
	template<typename InputIterator>
	Point approximateOneMedianRounds(InputIterator begin, InputIterator end, double eps, int rounds = 0);
	
	/**
	 * @brief Approximate k-median
	 * @param begin Point iterator
	 * @param end Point iterator
	 * @param eps Factor of approximation
	 */
	template<typename InputIterator>
	Point approximateOneMedian(InputIterator begin, InputIterator end, double eps);
	
private:
	template<typename InputIterator>
	std::unique_ptr<std::vector<Point>> constructSatisfyingSet(InputIterator begin, InputIterator end, double eps, size_t sizeOfRandomSample);
	
	template<typename InputIterator>
	std::unique_ptr<std::vector<Point>> constructSatisfyingSet(InputIterator begin, InputIterator end, double a, double b, double eps, size_t sizeOfRandomSample);
	
	std::unique_ptr<std::vector<Point>> constructGridSphere(std::vector<Point> const & basis, Point basePoint, double sideLength, double radius);	

	Metric<Point>* metric;
	Norm<Point>* norm;
	KMedian kmedian;
	EuclideanSpace space;
};

template<typename InputIterator>
Point KumarMedian::approximateOneMedianRounds(InputIterator begin, InputIterator end, double eps, int rounds)
{
	if(rounds == 0)
	{
		int dim = begin->getDimension();
		if(dim <= 5)
			rounds = 8;
		else if(dim <= 10)
			rounds = 12;
		else if(dim <= 20)
			rounds = 15;
		else if(dim <= 50)
			rounds = 16;
		else
			rounds = 18;
	}
	
	Point minPoint;
	double cost = 0;
	for(int i = 0; i < rounds; ++i)
	{
		Point tmpPoint(approximateOneMedian(begin, end, eps));
		std::unique_ptr<std::vector<Point>> samSet(Sampling::sampleWithoutReplacementFast<InputIterator, Point>(begin, end, 1000));
		double tmpCost(kmedian.cost(samSet->begin(), samSet->end(), tmpPoint));
		if(tmpCost < cost || i == 0)
		{
			cost = tmpCost;
			minPoint = tmpPoint;
		}
	}
	return minPoint;
}

template<typename InputIterator>
Point KumarMedian::approximateOneMedian(InputIterator begin, InputIterator end, double eps)
{
	// Theorem 5.7
	size_t sampleSize = 1;
	std::unique_ptr<std::vector<Point>> satSet(constructSatisfyingSet(begin, end, eps, sampleSize));
	std::unique_ptr<std::vector<Point>> samSet(Sampling::sampleWithoutReplacementFast<InputIterator, Point>(begin, end, 10));
	
	double cost = 0;
	Point median;
	for(size_t i = 0; i < satSet->size(); ++i)
	{
		double tmpcost = kmedian.cost(samSet->begin(), samSet->end(), (*satSet)[i]);
		if(tmpcost < cost || i == 0)
		{
			cost = tmpcost;
			median = (*satSet)[i];
		}
	}
	
	return median;
}

template<typename InputIterator>
std::unique_ptr<std::vector<Point>> KumarMedian::constructSatisfyingSet(InputIterator begin, InputIterator end, double eps, size_t sizeOfRandomSample)
{ // Theorem 5.4
	std::unique_ptr<std::vector<Point>> sample(Sampling::sampleWithoutReplacementFast<InputIterator, Point>(begin, end, std::ceil(1.0/eps+1.0)));
	double v = kmedian.cost(sample->begin()+1, sample->end(), (*sample)[0]);
	
	double a = v * std::pow(eps, 3) / 2;
	double b = v / eps;
	if(a < b)
	{
		std::unique_ptr<std::vector<Point>> set(constructSatisfyingSet(sample->begin()+1, sample->end(), a, b, eps, sizeOfRandomSample));
		set->push_back((*sample)[0]);
		return set;
	}
	else
	{
		std::unique_ptr<std::vector<Point>> set(new std::vector<Point>(sample->begin(), sample->begin()+1));
		return set;
	}
}

template<typename InputIterator>
std::unique_ptr<std::vector<Point>> KumarMedian::constructSatisfyingSet(InputIterator begin, InputIterator end, double a, double b, double eps, size_t sizeOfRandomSample)
{ // Lemma 5.3
	std::unique_ptr<std::vector<Point>> sampleVector(Sampling::sampleWithoutReplacementFast<InputIterator, Point>(begin, end, sizeOfRandomSample));
	int lowerBound(std::floor(std::log(eps/4.0*a)));
	int upperBound(std::ceil(std::log(b)));
	std::unique_ptr<std::vector<Point>> basis(space.orthonormalize(sampleVector->begin(), sampleVector->end()));
	
	std::unique_ptr<std::vector<Point>> satisfyingSet(new std::vector<Point>());
	for(auto it = sampleVector->begin(); it != sampleVector->end(); ++it)
	{
		for(int i = lowerBound; i <= upperBound; ++i)
		{
			double t = std::pow(2, i);
			std::unique_ptr<std::vector<Point>> subset = constructGridSphere(*basis, *it, (eps*t)/(4.0*sampleVector->size()), 2.0*t);
			satisfyingSet->insert(satisfyingSet->end(), subset->begin(), subset->end());
		}
	}
	
	return satisfyingSet;
}

#endif
