#include <memory>
#include <iostream>
#include <sstream>
#include <iomanip>
#include <vector>
#include <iterator>
#include <fstream>

#ifdef ERRORHANDLER
#include <cstdlib>
#include <stdexcept>
#include <execinfo.h>
#endif

#include <boost/algorithm/string.hpp>

#include "../probi-source/PKMedian.hpp"
#include "../probi-source/WeightedPoint.hpp"
#include "../probi-source/Point.hpp"
#include "../probi-source/ProbabilisticPoint.hpp"
#include "../probi-source/EuclideanMetric.hpp"
#include "../probi-source/EuclideanSquaredMetric.hpp"

#include "common.hpp"

using namespace std;

int main(int argc, char** argv)
{
#ifdef ERRORHANDLER
    std::set_terminate(handler);
#endif

    if (argc < 6)
    {
        std::cout << "Usage: k splitChar numOfPointsToMerge isWeighted centerFile dataFile" << std::endl;
        std::cout << "e.g. : 10 S 10 1 centers.dat data.dat" << std::endl;
        return 1;
    }

    // k argument
    int k = atoi(argv[1]);
    // SplitChar argument
    char splitChar = *argv[2];
    if (splitChar == 'S')
        splitChar = ' ';
    std::stringstream splitStream;
    splitStream << splitChar;
    std::string splitSeq(splitStream.str());
    // Number of points to merge
    int numOfPointsToMerge = atoi(argv[3]);
    // Weight flag
    int int_isWeighted = atoi(argv[4]);
    bool isWeighted = (int_isWeighted == 1);          
    // Center file
    std::fstream centerStream(argv[5], std::fstream::in);
    // Data file
    std::fstream dataStream(argv[6], std::fstream::in);

#ifdef KMEANS
    PKMedian pkmed([]()
    {
        return new EuclideanSquaredMetric();
    });
#else
    PKMedian pkmed([]()
    {
        return new EuclideanMetric();
    });
#endif
    
    std::vector<Point> centers;
    std::vector<ProbabilisticPoint> ppoints;
    double prob = 1 / double(numOfPointsToMerge);

    // Data dimension
    int dim = 0;
    std::function<bool(ProbabilisticPoint) > determineDimension = [&dim] (ProbabilisticPoint pp)
      {
          dim = pp[0].getDimension();
          return false;
      };
    dataStream.clear();
    dataStream.seekg(0);
    processStream(dataStream, determineDimension, splitSeq, numOfPointsToMerge, prob, isWeighted);

    // Read center file
    int realK = 0;
    std::function<bool(ProbabilisticPoint) > readCenters = [&centers, &realK] (ProbabilisticPoint pp)
      {
        centers.push_back(pp[0]);
        ++realK;
        return true;
      };
    centerStream.clear();
    centerStream.seekg(0);
    processStream(centerStream, readCenters, ",", 1, 1, false);

    // Read data file
    int n = 0;
    std::function<bool(ProbabilisticPoint) > readData = [&ppoints, &n] (ProbabilisticPoint pp)
      {
        ppoints.push_back(pp);
        ++n;
        return true;
      };
    dataStream.clear();
    dataStream.seekg(0);
    processStream(dataStream, readData, splitSeq, numOfPointsToMerge, prob, isWeighted);
    
    // Compute costs
    double costs = pkmed.weightedCost(ppoints.begin(), ppoints.end(), centers.begin(), centers.end());
    
    std::cout << "k                         : " << k << std::endl;
    std::cout << "Real number of centers    : " << centers.size() << std::endl;
    std::cout << "Dimension                 : " << dim << std::endl;
    std::cout << "Data size                 : " << n << std::endl;
    std::cout << std::endl;
    std::cout << "Costs                     : " << costs << std::endl;    

    return 0;
}

